<?php
 /**
 * Jamroom Developer Tools module
 *
 * copyright 2023 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrDeveloper_meta()
{
    return array(
        'name'        => 'Developer Tools',
        'url'         => 'developer',
        'version'     => '2.0.8',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Developer tools for working with modules and skins',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/932/developer-tools',
        'license'     => 'mpl',
        'requires'    => 'jrCore:6.5.12',
        'category'    => 'developer'
    );
}

/**
 * init
 */
function jrDeveloper_init()
{
    jrCore_register_module_feature('jrCore', 'javascript', 'jrDeveloper', 'jrDeveloper.js', 'admin');
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', jrCore_get_base_url() . "/modules/jrDeveloper/adminer.php", array('Database Admin', 'Browse your Database Tables - <b>carefully!</b>'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', jrCore_get_base_url() . "/modules/jrDeveloper/apc.php", array('Memory Cache Admin', 'View the local Memory Cache statistics and usage'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'clone_skin', array('Clone Skin', 'Save a copy of an existing skin to a new name'));
    if (jrCore_get_config_value('jrDeveloper', 'developer_prefix', false)) {
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'package_module', array('Package Module', 'Create a Module ZIP Package that can be uploaded to the Jamroom Marketplace'));
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'package_skin', array('Package Skin', 'Create a Skin ZIP Package that can uploaded to the Jamroom Marketplace'));
    }
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'query_stats', array('Database Query Counts', 'View SQL query counts and execution time'));
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'lang_verify', array('Language Log', 'Possible language string errors found in template files'));

    if (jrCore_get_config_value('jrDeveloper', 'enable_reset', 'off') == 'on') {
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'rebase_modules', array('Rebase Modules', 'Move marketplace modules back to their root folder and remove symlinks'));
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'rebase_skins', array('Rebase Skins', 'Move marketplace skins back to their root folder and remove symlinks'));
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'reset_categories', array('Reset Categories', 'Return each module to its default module category in the ACP'));
        jrCore_register_module_feature('jrCore', 'tool_view', 'jrDeveloper', 'reset_system', array('Reset System', 'Reset the system to the state of a fresh install'));
    }

    // Our default view for admins
    jrCore_register_module_feature('jrCore', 'default_admin_view', 'jrDeveloper', 'admin/tools');

    // Loader listeners
    jrCore_register_event_listener('jrCore', 'process_init', 'jrDeveloper_process_init_listener');
    jrCore_register_event_listener('jrCore', 'parsed_template', 'jrDeveloper_parsed_template_listener');
    jrCore_register_event_listener('jrCore', 'db_query_init', 'jrDeveloper_db_query_init_listener');
    jrCore_register_event_listener('jrCore', 'db_query_exit', 'jrDeveloper_db_query_exit_listener');
    jrCore_register_event_listener('jrCore', 'process_exit', 'jrDeveloper_process_exit_listener', 1000);

    // We have an event trigger
    jrCore_register_event_trigger('jrDeveloper', 'reset_system', 'Fired when the Reset System tool is run');

    return true;
}

//----------------------
// FUNCTIONS
//----------------------

/**
 * Get the version number for a local module
 * @param $mod string module
 * @return bool|string
 */
function jrDeveloper_get_local_module_version($mod)
{
    global $_mods;
    // Get version ON DISK if we can
    $vers = false;
    $_mta = @file(APP_DIR . "/modules/{$mod}/include.php");
    if ($_mta && is_array($_mta)) {
        $meta = false;
        foreach ($_mta as $line) {
            if (strpos(' ' . $line, "function") && strpos($line, '_meta()')) {
                $meta = true;
                continue;
            }
            if ($meta) {
                $line = trim(trim(str_replace(array('"', "'"), '', $line)), ',');
                $mkey = jrCore_string_field($line, 1);
                switch ($mkey) {
                    case 'version':
                        list(, $text) = explode('=>', $line);
                        if (isset($text) && strlen(trim($text)) > 0) {
                            $vers = trim($text);
                            break 2;
                        }
                        break;
                }
            }
        }
    }
    return ($vers) ? $vers : $_mods[$mod]['module_version'];
}

/**
 * Reset any configured opcode caches
 * @return bool
 */
function jrDeveloper_reset_opcode_caches()
{
    if (function_exists('apc_clear_cache')) {
        apc_clear_cache();
    }
    if (function_exists('xcache_clear_cache') && defined('XC_TYPE_PHP')) {
        $on = ini_get('xcache.admin.enable_auth');
        if ($on != 1 && $on != 'on') {
            @xcache_clear_cache(XC_TYPE_PHP, 0);
        }
        else {
            // [xcache.admin]
            // xcache.admin.enable_auth = Off
            // ; Configure this to use admin pages
            // ; xcache.admin.user = "mOo"
            // ; xcache.admin.pass = md5($your_password)
            // ; xcache.admin.pass = ""
            // See if we have been setup properly
            if (strlen(ini_get('xcache.admin.user')) > 0 && ini_get('xcache.admin.user') !== 'mOo') {
                @xcache_clear_cache(XC_TYPE_PHP, 0);
            }
        }
    }
    if (function_exists('opcache_reset')) {
        opcache_reset();
    }
    return true;
}

/**
 * Rebase/reset the modules or skins directory
 * @param $dir string one of "modules" or "skins"
 * @param string $delete on|off delete old version dirs
 * @return array|false
 */
function jrDeveloper_rebase_directory($dir, $delete = 'off')
{
    switch ($dir) {
        case 'modules':
            $tag = 'module';
            break;
        case 'skins':
            $tag = 'skin';
            break;
        default:
            return false;
    }
    // do stuff
    $base_dir = APP_DIR . '/' . $dir;
    $_dir     = glob($base_dir . '/*');
    $i        = 0;
    foreach ($_dir as $link) {
        if (is_link($link)) {
            // its a symlink, move it to its base dir.
            $module    = basename($link);
            $linked_to = readlink($link);
            if (strpos($linked_to, '-release-')) {

                // remove symlink
                if (unlink("{$base_dir}/{$module}")) {
                    // move newest release to the basedir.
                    if (!rename("{$base_dir}/{$linked_to}", "{$base_dir}/{$module}")) {
                        // We have a problem - recreate symbolic link or system will be in an unusable state
                        chdir($base_dir);
                        if (!symlink($linked_to, $module)) {
                            jrCore_set_form_notice('error', "Unable to rename {$tag} OR re-link old {$tag} {$module} - restore {$tag} directory via FTP");
                            jrCore_form_result();
                        }
                        jrCore_set_form_notice('error', "Unable to rename old {$tag} {$module} - check file permissions");
                        jrCore_form_result();
                    }
                }
                $i++;
            }
        }
    }

    // delete old versions.
    $x = 0;
    if ($delete == 'on') {
        $_dir = glob("{$base_dir}/*-release-*");
        foreach ($_dir as $link) {
            // remove it
            jrCore_delete_dir_contents($link, false);
            if (!rmdir($link)) {
                jrCore_set_form_notice('error', "Unable to delete old version directory for {$tag}: " . $link);
                jrCore_form_result();
            }
            $x++;
        }
    }

    // Reset smarty caches
    $_tmp = glob(APP_DIR . "/data/cache/*");
    if (is_array($_tmp)) {
        $_dirs = array();
        foreach ($_tmp as $path) {
            if (is_dir($path)) {
                $_dirs[] = basename($path);
            }
        }
        $_dirs = jrCore_trigger_event('jrCore', 'template_cache_reset', $_dirs); // "template_cache_reset" event trigger
        foreach ($_dirs as $dir) {
            if ($dir == 'jrImage') {
                // Image cache is handled by separate Image cache reset tool
                continue;
            }
            if (is_dir(APP_DIR . "/data/cache/{$dir}")) {
                jrCore_delete_dir_contents(APP_DIR . "/data/cache/{$dir}");
            }
        }
    }

    jrCore_delete_all_cache_entries();
    jrDeveloper_reset_opcode_caches();
    clearstatcache(true);
    return array($i, $x);
}

/**
 * Add A license header to a file
 * @param $type string type of license
 * @param $name string module name
 * @param $file string file
 * @param $license string license
 * @return bool
 */
function jrDeveloper_add_license_header($type, $name, $file, $license)
{
    global $_mods;
    if (!is_file($file)) {
        return false;
    }
    switch ($type) {
        case 'module':
            $_rep = array(
                'item_name'      => $_mods[$name]['module_name'],
                'item_directory' => $_mods[$name]['module_directory'],
                'item_type'      => 'module'
            );
            break;
        case 'skin':
            $_tmp = jrCore_skin_meta_data($name);
            $_rep = array(
                'item_name'      => $_tmp['name'],
                'item_directory' => $name,
                'item_type'      => 'skin'
            );
            break;
        default:
            jrCore_logger('CRI', "developer: invalid item type received - must be one of module,skin");
            return false;
    }
    $open = false;
    $temp = "<?php\n" . jrCore_parse_template("{$license}_header.tpl", $_rep, 'jrDeveloper');
    $_tmp = file($file);
    foreach ($_tmp as $line) {
        if (strpos($line, '<?php') === 0 && !$open) {
            $open = true;
            continue;
        }
        elseif ($open && strpos(trim($line), '/*') === 0) {
            $open = false;
            continue;
        }
        $temp .= rtrim($line) . "\n";
    }
    return jrCore_write_to_file($file, $temp);
}

/**
 * Get the query count data
 * @return array
 */
function jrDeveloper_get_query_count_data()
{
    $_out = array(
        'keys' => array(),
        'time' => array()
    );
    if ($_rt = apcu_cache_info()) {
        if (!empty($_rt['cache_list'])) {
            $pfx = jrCore_local_cache_get_key_prefix(false);
            foreach ($_rt['cache_list'] as $i) {
                if (strpos($i['info'], "{$pfx}:c:") === 0) {
                    $_out['keys']["{$i['info']}"] = apcu_fetch($i['info']);
                }
                elseif (strpos($i['info'], "{$pfx}:d:") === 0) {
                    $_out['time']["{$i['info']}"] = apcu_fetch($i['info']);
                }
            }
        }
    }
    return $_out;
}

/**
 * Reset the Query Count data in APCu
 * @return int
 */
function jrDeveloper_reset_query_count_data()
{
    $num = 0;
    $_rt = apcu_cache_info();
    if ($_rt && is_array($_rt) && isset($_rt['cache_list'])) {
        $pfx = jrCore_local_cache_get_key_prefix(false);
        foreach ($_rt['cache_list'] as $v) {
            if (strpos($v['info'], "{$pfx}:c:") === 0 || strpos($v['info'], "{$pfx}:d:") === 0) {
                if (apcu_delete($v['info'])) {
                    $num++;
                }
            }
        }
    }
    return $num;
}

//----------------------
// EVENT LISTENERS
//----------------------

/**
 * Track SQL query counts for performance
 * @param string $query SQL Query
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return string
 */
function jrDeveloper_db_query_init_listener($query, $_user, $_conf, $_args, $event)
{
    // Save this query if we are logging
    if ($_sum = jrCore_get_flag('jrdeveloper_query_sum')) {
        if (!is_array($_sum)) {
            $_sum = array();
        }
        $index = md5($query);
        if (!isset($_sum[$index])) {
            $_sum[$index] = array();
        }
        $now            = microtime(true);
        $_sum[$index][] = "{$now}:{$query}";
        jrCore_set_flag('jrdeveloper_query_sum', $_sum);
    }

    // Set our flag so we know a query is active - this prevents us
    // from running another query in our exit listener and
    // get stuck in a recursive query loop
    $save = true;
    if (jrCore_client_is_detached()) {
        // This is a background worker - don't worry about DELETE or INSERT or UPDATE
        if (strpos($query, 'SELECT') !== 0) {
            $save = false;
        }
    }
    // Don't report on alter table queries
    elseif (strpos($query, 'ALTER') === 0) {
        $save = false;
    }
    if ($save) {
        if (!jrCore_get_flag('jrdeveloper_active_query')) {
            jrCore_set_flag('jrdeveloper_active_query', $query);
        }
    }
    return $query;
}

/**
 * Track SQL query duration for performance
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrDeveloper_db_query_exit_listener($_data, $_user, $_conf, $_args, $event)
{
    // Are we keeping track of slow SQL queries?
    if ($query = jrCore_get_flag('jrdeveloper_active_query')) {
        if ($seconds = jrCore_get_config_value('jrDeveloper', 'slow_queries', 0)) {
            if ($seconds > 0 && isset($_data['query_ms']) && $_data['query_ms'] >= 250 && $_data['query_ms'] >= $seconds) {
                jrCore_logger('MIN', "developer: slow query took {$_data['query_ms']}ms", $query);
            }
        }
        jrCore_delete_flag('jrdeveloper_active_query');
    }

    // Are we tracking query counts?
    if (jrCore_get_config_value('jrDeveloper', 'query_counts', 'off') == 'on') {
        if (strpos(PHP_VERSION, '5.3') !== 0 && strpos($_args[0], 'SELECT') === 0) {
            if ($_tr = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 10)) {
                if (isset($_tr[0])) {
                    $_fn = false;
                    foreach ($_tr as $v) {
                        if (isset($v['file']) && !strpos($v['file'], 'jrCore')) {
                            $_fn = $v;
                            break;
                        }
                    }
                    if (is_array($_fn)) {
                        $file = str_replace(APP_DIR . '/', '', $_fn['file']);
                        $key  = md5("{$file}:{$_fn['line']}");
                        jrCore_increment_local_cache_key("c:{$key}:{$file}:{$_fn['line']}");
                        if ($_data['query_ms'] > 0) {
                            jrCore_increment_local_cache_key("d:{$key}", $_data['query_ms']);
                        }
                    }
                }
            }
        }
    }

    return $_data;
}

/**
 * Add Template name as an HTML comment if enabled
 * @param string $html incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return string
 */
function jrDeveloper_parsed_template_listener($html, $_user, $_conf, $_args, $event)
{
    if (jrCore_get_config_value('jrDeveloper', 'template_debug', 'off') == 'on') {
        // We only show template in source IF the template is:
        // - a skin template
        // - a module ITEM template
        $show = false;
        $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
        if ($_args['jr_template_directory'] == $skin) {
            $show = true;
        }
        elseif (jrCore_module_is_active($_args['jr_template_directory'])) {
            // This is a MODULE.  Only show in source if template begins with "item_" or is "index.tpl"
            if ($_args['jr_template'] == 'index.tpl' || strpos($_args['jr_template'], 'item_') === 0) {
                $show = true;
            }
        }
        if ($show) {
            $tpl  = "{$_args['jr_template_directory']}/{$_args['jr_template']}";
            $html = "\n<!-- BEGIN {$tpl} -->\n{$html}\n<!-- END {$tpl} -->\n";
        }
    }
    return $html;
}

/**
 * Turn on PHP logging if developer mode is on
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrDeveloper_process_init_listener($_data, $_user, $_conf, $_args, $event)
{
    // Turn on error logging if developer mode is on
    if (jrCore_get_config_value('jrDeveloper', 'developer_mode', 'off') == 'on') {
        error_reporting(E_ALL);
    }
    if (jrCore_get_config_value('jrDeveloper', 'query_sum', 'off') == 'on') {
        jrCore_set_flag('jrdeveloper_query_sum', 1);
    }
    return $_data;
}

/**
 * Save cumulative Query info
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrDeveloper_process_exit_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if ($_sum = jrCore_get_flag('jrdeveloper_query_sum')) {
        jrCore_delete_flag('jrdeveloper_query_sum');
        $_out = array();
        foreach ($_sum as $s) {
            $_out[] = "{$_post['_uri']}:" . getmypid() . ':' . str_replace("\n", ' ', $s[0]);
        }
        if (count($_out) > 0) {
            jrCore_write_to_file(APP_DIR . '/data/logs/sql_query_log', "\n" . implode("\n", $_out), 'append');
        }
    }
    return $_data;
}

/**
 * Export any custom Form Designer fields to a module
 * @param $module string Module to export
 * @param $path string full path to module
 * @return bool|string
 */
function jrDeveloper_export_form_designer_fields($module, $path)
{
    $_views = array('update', 'create');
    foreach ($_views as $view) {
        $tmp = jrCore_get_designer_form_fields($module, $view);
        if (is_array($tmp)) {
            $_fields[$view] = $tmp;
        }
    }
    if (isset($_fields) && is_array($_fields)) {
        $fullpath = $path . '/custom_form_fields.json';
        jrCore_write_to_file($fullpath, json_encode($_fields));
        return $fullpath;
    }
    return false;
}

/**
 * Create lang file to include custom lang strings
 * @param $module string Module to export lang files for
 * @param $path string full path to file
 * @return bool
 */
function jrDeveloper_export_lang_strings($module, $path)
{
    $tbl = jrCore_db_table_name('jrUser', 'language');
    $req = "SELECT * FROM {$tbl} WHERE lang_code = 'en-US' AND lang_module = '{$module}'";
    $_rt = jrCore_db_query($req, 'NUMERIC');

    if (isset($_rt) && is_array($_rt)) {
        $temp    = '';
        $started = false;
        $_tmp    = file($path);
        foreach ($_tmp as $line) {
            if (!$started) {
                $temp .= rtrim($line) . "\n";
            }
            else {
                break;
            }
            if (preg_match('/or exit/', $line)) {
                $started = true; // inside the writable area.
            }

        }
        // got the headers for the lang file, now write the current state of the DB as the contents.
        foreach ($_rt as $_l) {
            $temp .= '$lang[\'' . $_l['lang_key'] . '\'] =  \'' . jrCore_entity_string($_l['lang_text']) . '\';' . "\n";
        }
        jrCore_write_to_file($path, $temp);
    }
    return true;
}

/**
 * evaluates whether one module version is higher or lower than another.
 * @param $a string version 1
 * @param $b string version 2
 * @return string
 */
function jrDeveloper_ver_compare($a, $b)
{
    switch (version_compare($a, $b)) {
        case -1:
            return 'less';
        case 1:
            return 'greater';
    }
    return 'equal';
}

/**
 * Verify a language string
 * @param string $current
 * @param string $default
 * @param string $module
 * @param int $id
 * @param array $_settings
 * @param string $tpl
 * @param string $uri
 * @return bool
 */
function jrDeveloper_verify_lang($current, $default, $module, $id, $_settings, $tpl, $uri)
{
    if ($current == $default) {
        return true;
    }
    if ($_settings['code'] != 'en-US') {
        return true;
    }
    $tpl = substr($tpl, 0, 100); // truncate the template code passed in

    $tbl = jrCore_db_table_name('jrUser', 'language');
    $req = "SELECT lang_default FROM {$tbl} WHERE lang_module = '{$module}' AND lang_key = '{$id}' AND lang_code = '{$_settings['code']}'";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt) && isset($_rt['lang_default'])) {
        if (strtolower($_rt['lang_default']) !== strtolower($default) && jrCore_checktype($id, 'number_nz')) {
            // log the skin/db mismatch
            $_sv = array(
                'log_module'     => $module,
                'log_skin_value' => $default,
                'log_db_value'   => $_rt['lang_default'],
                'log_id'         => $id,
                'log_tpl'        => $tpl,
                'log_uri'        => $uri,
                'log_lng'        => $_settings['code']
            );
            jrCore_db_create_item('jrDeveloper', $_sv);
        }
    }
    return true;
}
